/**************************************************************\
 *  Prototipi
\**************************************************************/
void codec_out(short sample);
void codec_init(void);
void init_HWI(void);
void emif_init(void);
short sineDelayGen(void);

/**************************************************************\
 *  Include
\**************************************************************/
#include <c6x.h>           // definizioni del compilatore C6000

#include <csl.h>           // header CSL
#include <csl_irq.h>
#include <csl_mcbsp.h>

#include <bsl.h>           // header BSL
#include <bsl_ad535.h>    
#include <bsl_suggestions.h>

#include "chorus.h"	   // costanti che servono per l'effetto

/**************************************************************\
 *  Variabili globali
\**************************************************************/

    /**********************************************************\
     *  AD535 Handle and Config structure
    \**********************************************************/
    AD535_Handle hAD535;                                       
    AD535_Config my_AD535_Config = {                           
        AD535_LOOPBACK_DISABLE,                                
        AD535_MICGAIN_OFF,                                     
        AD535_GAIN_0DB,                                        
        AD535_GAIN_0DB                                         
    };                                                         

    /**********************************************************\
     * Buffer circolare
    \**********************************************************/
    short input_buf[BUFFER_SIZE];	// 240 è il minimo
    // indice del buffer
    short input_idx = 0;

/**************************************************************\
 *  Main Routine
\**************************************************************/
void main()
{
    int i;
    CSL_init();              // Inizializza la libreria CSL
    BSL_init();              // Inizializza la libreria BSL
    codec_init();
    init_HWI();
 
    AD535_write(hAD535, 0);  /* Scrivo qualcosa per far iniziare
				mcbsp a trasmettere interrupt */

    // Inizializzo il buffer prima di procedere (azzerandolo)
    for (i=0; i<BUFFER_SIZE; i++)
      input_buf[i]=0;

    while(1);
}

/*************************************************************\
 *	Inizializza gli interrupt hardware. 
 **************************************************************
 *  Il codec AD535 è collegato fisicamente all'McBSP0. Quindi,
 *  il nostro sistema usa l'interrupt McBSP0 di trasmissione
 *  (XINT0) per segnalare quando scrivere il prossimo campione
 *  in uscita.
 *  (XINT0 è stato mappato sull'interrupt 9 (HWI9) tramite
 *   il tool di configurazione)
\*************************************************************/
void init_HWI(void)
{
 	// Attiva tutti gli interrupt
	IRQ_globalEnable();

	// Attiva l'interrupt McBSP0 di trasmissione
	IRQ_enable(IRQ_EVT_XINT0);
}

/**********************************************************\
 *	Codec Functions
\**********************************************************/
void codec_init()
{
	/* Uso le routine BSL per l'open, il reset ed il config dell'AD535	*\
	 * Parte della routine AD535_open() apre e configura l'McBSP0, visto	*
	\* che tutte le comunicazioni col codec AD535 sono gestite via McBSP0.	*/
	hAD535 = AD535_open(AD535_localId);	// Apre l' AD535 e ne prende l'handle
	AD535_reset(hAD535);			// Resetta l' AD535                     
	AD535_config(hAD535, &my_AD535_Config);	// Configura l' AD535                 
	
	/**************************************************\
	 *	Imposto un bit nel registro McBSP in modo
	 *	tale da farlo continuare a funzionare anche
	 *	quando il debugger smette di funzionare.
         *      Questo evita che si corrompe il codec
	\**************************************************/
	MCBSP_setfree(0);
}

/******************************************************\
 * Mette in uscita al codec AD535 il campione passato
\******************************************************/
void codec_out(short sample) 
{
	AD535_HWI_write(hAD535, sample);
}

/******************************************************\
 * Preleva il campione disponibile dal codec AD535
\******************************************************/
short codec_in() 
{ 
	return AD535_HWI_read(hAD535);
}

/******************************************************\
 * Si è verificato l'interrupt
 * (è disponibile un nuovo campione)
\******************************************************/
void XINT0_HWI() {
  // contiene i campioni ritardati
  float delayed_sample;

  // contiene il ritardo espresso in numero di campioni
  short samples_delay;

  // contiene il campione corrente
  short current_sample = codec_in();

  // Metto nel buffer circolare il campione corrente
  input_buf[input_idx] = current_sample;

  // ritardo LFO in questo istante (in numero di campioni)
  samples_delay = sineDelayGen();
  
  // Prelevo il campione ritardato (rispetto al corrente)
  // dal buffer circolare e lo amplifico di CHORUS_GAIN.
  // Utilizzo la AND per realizzare un indice circolare,
  // per valori negativi ( es. : 0 - 160 = 255 - 159 = 94 )
  delayed_sample = (CHORUS_GAIN * input_buf[(input_idx-samples_delay) & 255]);

  // Al campione ritardato, gli sommo un altro campione
  // ritardato di 1ms (8 campioni) in meno,
  // con un guadagno attenuato di 0.3
  delayed_sample += ((CHORUS_GAIN-0.3) * input_buf[(input_idx-(samples_delay-8)) & 255]);

  // Ai due campioni ritardati, gli sommo un altro campione
  // ritardato di 2ms (16 campioni) in meno,
  // con un guadagno attenuato di 0.6
  delayed_sample += ((CHORUS_GAIN-0.6) * input_buf[(input_idx-(samples_delay-16)) & 255]);

  // Metto in uscita il campione corrente sommato a quelli ritardati,
  // tutti amplificati di VOLUME (per rendere più udibili i suoni)
  codec_out(VOLUME * (current_sample+((short) delayed_sample)));

  // indice circolare che realizza l'accesso circolare al buffer
  input_idx = (input_idx + 1) & 255;
}
