;==========================================================================;
;=INFORMATION           PROGRAM         ; octave 6 generator             =;
;=============          FILE NAME       ; octave6.asm                        =;
;=                      INCLUDE FILES   ; no include files                =;
;=                      PROCESSOR       ; TMS320C5x                       =;
;=                      AUTHOR          ; mba uh     =;
;=                      DATE/VERSION    ; 4/97  / Ver 1.00            =;
;==========================================================================;
;=DESCRIPTION                                                             =;
;=============                                                            =;
;=                                                                        =;
;= octave6:        A program that initialises the TMS320C50 DSK processor    =;
;=              and AIC codec. At a sampling rate of 8kHz, the DSK        =;
;=              performs a serial inpu port dummy read.                   =;
;=                                                                        =;
;=              A digital oscillator program generates a series of sine   =;
;=              wave constituting an octave    =;
;=                                                               =;
;=              no harmonics are used only pure sine waves                =;
;=              The frequencies of the sine waves can be altered by changing =;
;=              the coefficent values within the data table. =;
;=                                                                        =;
;=                                                                        =;
;==========================================================================;
;=SPECIFICATION         FREQUENCY       ; 8kHz                            =;
;===============        FILTER          ; not present                     =;
;=                      MISC            ; TLC320C40 initialised           =;
;==========================================================================;
	.mmregs
;==========================================================================;
;=SETTINGS                                                                =;
;==========================================================================;
	.ds     1000h           ; Assemble to data memory
	.data
TA      .word   17              ; TA AIC value - Fcut = 8 KHz
RA      .word   17              ; RA AIC value - Fcut = 8 KHz
TB      .word   37              ; TB AIC value - Fs = 2*Fcut
RB      .word   37              ; RB AIC value - Fs = 2*Fcut
AIC_CTR .word   8h              ; |LP xx G1 G0 | SY AX LB BP|           
				; +------------+------------+            
				;  |     GAIN    |  |  |  +-- BP Filter   
				;  |     Synch --+  |  +----- Loopback     
				;  |     Auxin -----+                    
				;  + (sinx)/x filter 
;-------------------------------------------;
; note table
; fn = f note/f sampling
; Y/2 in Q15 format
; A/2 = cos(fn*360) in Q15 format
;-------------------------------------------;

do                                  ;880Hz
            .word   10453          ; Initial Y =  [sin(fn*360)/2] Q15
            .word   0           ; Secondary Y value
docoef               
             .word   25264    ; cos(fn*360) in Q15
re
             .word    11469    ; 987Hz
             .word     0
recoef
              .word    23396   ; coeff value
mi
              .word   11993    ;1046Hz
              .word    0
micoef
              .word   22315      ;coeff
fa
            .word   13074          ; 1174Hz
            .word   0           ;
facoef               
             .word   19792    ; 
sol
             .word     14090    ; 1318Hz
             .word     0
solcoef
              .word    16712   ; 
la
              .word   14582    ;1397Hz
              .word    0
lacoef
              .word   14942      ;
si
              .word   15466      ; 1568Hz
              .word    0
sicoef
              .word   10912      ;
;---------------------------------------;
;-- end of note table----------------;

;---------------------------------------;
TIME     .word   1000h    ; initial time, 0.4 seconds
; to change time change this value and reload value in TX INT
;---------------------------------------;
;==========================================================================;
;=INTERRUPT SETTINGS                                                      =;
;==========================================================================;
	.ps     080ah           ; Assemble to program memory
RINT    b       RX              ; Branch to RX on receive interrupt
XINT    b       TX              ; Branch to TX on transmit interrupt
;==========================================================================;
;=PROCESSOR INITIALISATION                                                =;
;==========================================================================;
	.ps 0a00h               ; Assemble to program memory
	.entry                  ; Mark program entry point
	.text                   ; Text
START   setc    INTM            ; Disable interrupts
	ldp     #0              ; Set data page pointer to page zero
	splk    #0836h,PMST     ; Write 16 bit pattern to PMST register
	lacc    #0              ; Load accumulator with number zero
	;-----------------------;
	samm    CWSR            ; Set software wait state to zero
	samm    PDWSR           ; Set software wait state to zero
	splk    #022h,IMR       ; Using XINT syn TX & RX
	call    AIC             ; Initialize AIC and enable interrupts
	;-----------------------;
	clrc    OVM             ; Overflow mode is set to zero
	spm     0                 ; Product shift mode is set to zero
              clrc   SXM
              lar      ar1,#do          ;point to first note
              lar      ar2,#docoef    ;point to first coefficient
              lar      ar3,#TIME     ;point to note length

	splk    #022h,IMR       ; Mask interrupts
	clrc    INTM            ; Enable interrupts

;==========================================================================;
;=MAIN PROGRAM                                                            =;
;==========================================================================;
WAIT
              nop                     ; Wait until interrupt
	nop                     
              nop                     ; No operation
	b       WAIT            ; Branch to WAIT
;---- end of main program ------; Comment line

;==========================================================================;
;= TRANSMIT INTERRUPT SERVICE ROUTINE                                     =;
;==========================================================================;

TX
              call  DELAY      ;time a note is played for
	sub   #1               ; end?
              bcnd  nochange,NEQ  ;if at end relaod timer counter
              lacc   #1000h   ;reload timer counter
              mar  *,ar3        
              sacl   *              ;store time for decrement 
              
              lamm   ar2          ; check where we are
              sub   #docoef,0     ;is it DO
              bcnd   setre,EQ    ; if DO set to RE
              
              lamm    ar2          
              sub  #recoef,0       ;is it RE
              bcnd    setmi,EQ   ;if RE set MI

              lamm    ar2
              sub  #micoef,0      ; MI?
              bcnd    setfa,EQ    ; if MI set FA

              lamm    ar2
              sub  #facoef,0         ; FA?
              bcnd    setsol,EQ    ; if FA set SOL

              lamm    ar2
              sub  #solcoef,0        ; SOL?
              bcnd    setla,EQ       ;if SOL set LA

              lamm    ar2
              sub  #lacoef,0          ;LA?
              bcnd    setsi,EQ       ;if LA set SI


setdo              	

              lar  ar1,#do            ;if none of the above
	lar  ar2,#docoef       ; set DO
              b  newfreq              ; start again

setre

              lar  ar1,#re              ;point to RE in note table
	lar  ar2,#recoef        ; point to RE coefficient in note table
              b  newfreq

setmi
	lar  ar1,#mi             ;point aux regs to MI
	lar  ar2,#micoef
              b  newfreq


setfa
	lar  ar1,#fa            ; aux regs to FA
	lar  ar2,#facoef
              b  newfreq


setsol
	lar  ar1,#sol           ;aux regs to SOL
	lar  ar2,#solcoef
              b  newfreq

setla
	lar  ar1,#la             ; aux regs to LA
	lar  ar2,#lacoef
              b  newfreq


setsi
	lar  ar1,#si              ;aux regs to si
	lar  ar2,#sicoef
              b  newfreq

nochange
newfreq           
;--------------------------------;
; note generation
; ar1 --> Y, ar2 --> coeff
; ar1 moves from Y to Y1
;--------------------------------;
	zap                     ; Clear acc and product register
	;-----------------------;
              ; acc = -B*Y1
              ;----------------------;
	mar  *,ar1       ;make ar1 current
	mar  *+,ar1     ;point to next value Y1
	lacc   *-,15,ar2    ; Load acc with Y1 with shift of 15,ar1->Y
	neg                      ; Negate accumulator
              ;-----------------------;
              ;  treg0 = coef
              ;  ar1 --> Y
              ;  preg = coef*Y
              ;-----------------------;
	 lt     *,ar1        ;load treg0=coeff, make ar1 current
              mpy    *           ;coeff*Y & place in preg, ar1 current
              dmov   *           ;Y->Y1 for next time round
	apac                    ; Add product register to acc
	apac                    ; all coeffs halved
              sach   *,1            ; out of Q15
	;-----------------------;
              lacc   *                 ;load sample to send
	ldp     #0              ; load data page zero
	and     #0FFFCh         ; Clear lower two (status) bit from acc
	samm    DXR             ; Write lower acc to DXR
	rete                    ; Return to main program

;==========================================================================;
;=RECEIVE INTERRUPT SERVICE ROUTINE                                       =;
;==========================================================================;
RX      
	rete                    ; RX INT not used
;==========================================================================;
;=BOARD INITIALISATION                                                    =;
;==========================================================================;
AIC     splk    #20h,TCR        ; To generate 10 MHz from Tout
	splk    #01h,PRD        ; Load period counter with 1
	mar     *,AR0           ; Modify AR pointer (used with GREG)
	lacc    #0008h          ; Load acc with 08h
	sacl    SPC             ; Store acc in SPC
	lacc    #00C8h          ; Load acc with 0C8h
	sacl    SPC             ; Set and reset SPC register
	lacc    #080h           ; Initialise 8000h to FFFFh as global memory
	sach    DXR             ; Store high acc to DXR
	sacl    GREG            ; Store to global memory register
	lar     AR0,#0FFFFh     ; Set AR0 register
	rpt     #10000          ; Set repeat counter
	lacc    *,0,AR0         ; Access global memory
	sach    GREG            ; Disable global memory
	;-----------------------;
	ldp     #TA             ; Load data page TA
	setc    SXM             ; Set SXM for sign extension mode
	lacc    TA,9            ; Load accumulator with TA data
	add     RA,2            ; Load accumulator with RB data
	call    AIC2ND          ; Call function AIC2ND
	;-----------------------;
	ldp     #TB             ; Load data page TB
	lacc    TB,9            ; Load accumulator with TB data
	add     RB,2            ; Load accumulator with RB data
	add     #02h            ; Add 10b to set status bits
	call    AIC2ND          ; Call function AIC2ND
	;-----------------------;
	ldp     #AIC_CTR        ; Load data page AIC_CTR
	lacc    AIC_CTR,2       ; Initialized control register
	add     #03h            ; Add 11b to set status bits
	call    AIC2ND          ; Call function AIC2ND
	ret                     ; Return from call
;-------------------------------;
AIC2ND  ldp     #0              ; Load data page zero
	sach    DXR             ; Store upper acc to DXR
	clrc    INTM            ; Enable interrupts
	idle                    ; Idle until interrupt - TINT
	add     #6h,15          ; Set 2 LSBs of upper acc high
	sach    DXR             ; Store upper acc to DXR
	idle                    ; Idle until interrupt - TINT
	sacl    DXR             ; Store upper acc to DXR
	idle                    ; Idle until interrupt - TINT
	lacl    #0              ; Store lower acc to DXR - flushing
	sacl    DXR             ; make sure the word got sent
	idle                    ; Idle until interrupt - TINT                   
	setc    INTM            ; Disable interrupts
	ret                     ; Return from call
;-------------------------------;

;==========================================================================;
;=TONE LENGTH ROUTINE                                         =;
;==========================================================================;
; time that a note is played, decrements TIME constant
; IN: AR3 --> TIME
; OUT: finished acc=1, not yet acc=2
DELAY
             mar  *,ar3    ; make ar3 current
             lacc  *          ;load count value ->acc
             sub  #1         ; decrement one
             sacl  *          ; store
             bcnd   carryon,GT  ; if not finished, carry on

              lacc #1         ; finished acc=1
              b  finish

carryon
              lacc #2      ; acc=2
finish      ret             ;acc=1     ;return with parameter in acc

;--------------------------------;

                      
	.end                    ; End of program






