#if !defined(NO_RTA)
#include <std.h>
#include <log.h>
#endif
#include <string.h> /* for memset */

#include "dtmf.h"

static const int16 sine_table[] = {
   804,   1608,   2411,   3212,   4011,   4808,   5602,   6393, 
  7180,   7962,   8740,   9512,  10279,  11039,  11793,  12540, 
 13279,  14010,  14733,  15447,  16151,  16846,  17531,  18205, 
 18868,  19520,  20160,  20788,  21403,  22006,  22595,  23170, 
 23732,  24279,  24812,  25330,  25833,  26320,  26791,  27246, 
 27684,  28106,  28511,  28899,  29269,  29622,  29957,  30274, 
 30572,  30853,  31114,  31357,  31581,  31786,  31972,  32138, 
 32286,  32413,  32522,  32610,  32679,  32729,  32758,  32767, 
 32758,  32729,  32679,  32610,  32522,  32413,  32286,  32138, 
 31972,  31786,  31581,  31357,  31114,  30853,  30572,  30274, 
 29957,  29622,  29269,  28899,  28511,  28106,  27684,  27246, 
 26791,  26320,  25833,  25330,  24812,  24279,  23732,  23170, 
 22595,  22006,  21403,  20788,  20160,  19520,  18868,  18205, 
 17531,  16846,  16151,  15447,  14733,  14010,  13279,  12540, 
 11793,  11039,  10279,   9512,   8740,   7962,   7180,   6393, 
  5602,   4808,   4011,   3212,   2411,   1608,    804,      0, 
  -804,  -1608,  -2411,  -3212,  -4011,  -4808,  -5602,  -6393, 
 -7180,  -7962,  -8740,  -9512, -10279, -11039, -11793, -12540, 
-13279, -14010, -14733, -15447, -16151, -16846, -17531, -18205, 
-18868, -19520, -20160, -20788, -21403, -22006, -22595, -23170, 
-23732, -24279, -24812, -25330, -25833, -26320, -26791, -27246, 
-27684, -28106, -28511, -28899, -29269, -29622, -29957, -30274, 
-30572, -30853, -31114, -31357, -31581, -31786, -31972, -32138, 
-32286, -32413, -32522, -32610, -32679, -32729, -32758, -32768, 
-32758, -32729, -32679, -32610, -32522, -32413, -32286, -32138, 
-31972, -31786, -31581, -31357, -31114, -30853, -30572, -30274, 
-29957, -29622, -29269, -28899, -28511, -28106, -27684, -27246, 
-26791, -26320, -25833, -25330, -24812, -24279, -23732, -23170, 
-22595, -22006, -21403, -20788, -20160, -19520, -18868, -18205, 
-17531, -16846, -16151, -15447, -14733, -14010, -13279, -12540, 
-11793, -11039, -10279,  -9512,  -8740,  -7962,  -7180,  -6393, 
 -5602,  -4808,  -4011,  -3212,  -2411,  -1608,   -804,      0 
};

/* return integer part of Q7 format */
#define Q7_INTEGER_PART(a) ((a & 0x7f80) >> 7)
#define Q7_ROUND(a) (a + 0x0040)

/* step size in Q7 format */
static uns16 stepSize[] = 
{ 
	2855,  /* 697 Hz */
	3154,  /* 770 Hz */
	3490,  /* 852 Hz */
	3854,  /* 941 Hz */
	4952,  /* 1209 Hz */
	5472,  /* 1336 Hz */
	6050,  /* 1477 Hz */
	6689   /* 1633 Hz */
};

/* constants for index into step size array */
typedef enum {
    F697,
    F770,
    F852,
    F941,
    F1209,
    F1336,
    F1477,
    F1633
} FREQ;

/* Table of digit to freq for row frequency */
static char rowTable[] = {
    0, /* invalid digit */
    F697,
    F697,
    F697,
    F770,
    F770,
    F770,
    F852,
    F852,
    F852,
    F941,
    F941,
    F941,
    F697,
    F770,
    F852,
    F941
};

/* Table of digit to freq for col frequency */
static char colTable[] = {
    0, /* invalid digit */
    F1209,
    F1336,
    F1477,
    F1209,
    F1336,
    F1477,
    F1209,
    F1336,
    F1477,
    F1336,
    F1209,
    F1477,
    F1633,
    F1633,
    F1633,
    F1633
};

/* phase delta for each tone. */
static uns16 rowDelta, colDelta;

/* current phase of tones */
static uns16 rowPhase, colPhase;

/* Number of tone samples left to generate. If this is zero then output
   should be silence */
static uns16 sampCount;

/* 1 if rowFreq and colFreq are valid. */
static int valid = 0;

/* Spec is 45-55ms of tone + 100 ms - tone of silence */
#define NUM_TONE_SAMPS 4000 /* 50 ms of tone */

/**
 * Set the frequency and enable the oscilator. If digit is not in the
 * range 0 < digit <= 16 then fill buf will write zeros into the
 * buffer. This should not be called more often then every 100 ms by
 * spec.
 * */
void
DTMF_setOutputDigit(int digit)
{
    /* stop oscilator */
    valid = 0; /* assumes DTMF_fillBuf always higher priority than this fxn; */

    if (digit > 0 && digit <= 16) {
	/* set step size. */
	rowDelta = stepSize[rowTable[digit]];
	colDelta = stepSize[colTable[digit]];
	/* clear phase accumulator */
	rowPhase = 0;
	colPhase = 0;

	/* and count */
	sampCount = NUM_TONE_SAMPS;

	/* restart osc */
	valid = 1;
    } 
}

/**
 * Fill a buffer of size with DTMF at the frequences specified in a
 * previous call to setFreq. If an invalid digit was given to
 * DTMF_setOutputDigit, if DTMF_setOutputDigit has not been called, or
 * the tone has been completed, then this buffer will be zeroed.
 * */        
static int16 *bufHead; /* for debug only */
void
DTMF_fillBuf(int16 *buf, int size)
{
    int i;
    int16 *p = bufHead = buf;

    if (valid == 1 && sampCount > 0) {
	for (i = 0; i < size && sampCount > 0; i++) {
	    int16 rowSamp, colSamp, sum;

	    rowSamp = sine_table[Q7_INTEGER_PART(Q7_ROUND(rowPhase))];
	    rowPhase += rowDelta;
	    colSamp = sine_table[Q7_INTEGER_PART(Q7_ROUND(colPhase))];
	    colPhase += colDelta;
		
	    /* combine tones for output */
	    /* gel programmable gain? */
	    sum = ((rowSamp >> 1) + (colSamp >>1)); // >> 2;
	    *p++ = sum;
	    sampCount--;
	}
	if (i < size) {
	    /* we hit the sampCount limit */
	    memset(p, 0, (size - i) * sizeof(int16));
	}
    } else {
	memset(p, 0, size * sizeof(int16));
    }
}

/**
 * Sum into a buffer of size with DTMF at the frequences specified in a
 * previous call to setFreq. If an invalid digit was given to
 * DTMF_setOutputDigit, or if DTMF_setOutputDigit has not been called, or
 * the tone has been completed, then return.
 * */        
void
DTMF_sumBuf(int16 *buf, int size)
{
    int i;
    int16 *p = bufHead = buf;

    if (valid == 1 && sampCount > 0) {
	for (i = 0; i < size && sampCount > 0; i++) {
	    int16 rowSamp, colSamp, sum;

	    rowSamp = sine_table[Q7_INTEGER_PART(Q7_ROUND(rowPhase))];
	    rowPhase += rowDelta;
	    colSamp = sine_table[Q7_INTEGER_PART(Q7_ROUND(colPhase))];
	    colPhase += colDelta;
		
	    /* combine tones for output */
	    /* gel programmable gain? */
	    sum = ((rowSamp >> 1) + (colSamp >>1)) >> 1;
	    *p++ += sum;
	    sampCount--;
	}
    }
}

