/* ======================================================================= */
/* TEXAS INSTRUMENTS, INC.                                                 */
/*                                                                         */
/* NAME                                                                    */
/*     DSPF_dp_vecrecip -- Double Precision vector reciprocal                   */
/*                                                                         */
/*  USAGE                                                                   */
/*                                                                          */
/*      This routine is C callable, and has the following C prototype:      */
/*                                                                          */
/*      void  DSPF_dp_vecrecip(const double *x,                                  */
/*                        double * restrict r,                              */
/*                        int n                                             */
/*                       )                                                  */
/*                                                                          */
/*             x        :  Pointer to input array                           */
/*             r        :  Pointer to output array                          */
/*             n        :  Number of elements in array                      */
/*                                                                          */
/*  DESCRIPTION                                                             */
/*                                                                          */
/*      The DSPF_dp_vecrecip module calculates the reciprocal of each element    */
/*  in the array x and returns the output in array r. It uses 3             */
/*  iterations of the Newton-Raphson method to improve the accuracy         */
/*  of the output generated by the RCPDP instruction of the C67x.           */
/*  Each iteration doubles the accuracy. The initial output generated       */
/*  by RCPDP is 8 bits.So after the first iteration it is 16 bits and       */
/*  after the second it is the  23 bits and after third it is full 52 bits. */
/*  The formula used is:                                                    */
/*                                                                          */
/*               r[n+1] = r[n](2 - v*r[n])                                  */
/*                                                                          */
/*      where v = the number whose reciprocal is to be found.               */
/*      x[0], the seed value for the algorithm, is given by RCPDP.          */
/*                                                                          */
/*  TECHNIQUES                                                              */
/*                                                                          */
/*      1. The inner loop is unrolled four times to allow calculation of    */
/*         four reciprocals in the kernel. However the stores are executed  */
/*         conditionally to allow 'n' to be any number > 0.                 */
/*                                                                          */
/*      2. Register sharing is used to make optimal use of available        */
/*         registers.                                                       */
/*                                                                          */
/*  ASSUMPTIONS                                                             */
/*                                                                          */
/*      1. There are no alignment requirements.                             */
/*                                                                          */
/*  C CODE                                                                  */
/*                                                                          */
/*      This is the C equivalent of the Assembly Code without               */
/*      restrictions.                                                       */
/*                                                                          */
/*    void DSPF_dp_vecrecip(const double* x, double* restrict r, int n)          */
/*    {                                                                     */
/*     int i;                                                               */
/*     for(i = 0; i < n; i++)                                               */
/*        r[i] = 1 / x[i];                                                  */
/*      }                                                                   */
/*                                                                          */
/*  NOTES                                                                   */
/*                                                                          */
/*      1. Endian: This code is LITTLE ENDIAN.                              */
/*      2. Interruptibility: This code is interrupt tolerant                */
/*         but not interruptible.                                           */
/*                                                                          */
/*  CYCLES                                                                  */
/*                                                                          */
/*     78*ceil(n/4) + 24                                                    */
/*     eg. for n = 54, cycles = 1116                                        */
/*                                                                          */
/*  CODESIZE                                                                */
/*                                                                          */
/*     448 bytes                                                            */
/* ------------------------------------------------------------------------ */
/*            Copyright (c) 2004 Texas Instruments, Incorporated.           */
/*                           All Rights Reserved.                           */
/* ======================================================================== */
#ifndef DSPF_DP_VECRECIP_H_
#define DSPF_DP_VECRECIP_H_ 1

void  DSPF_dp_vecrecip(const     double *x,
                  double * restrict r,
                  int n
                 );

#endif
/* ======================================================================== */
/*  End of file:  DSPF_dp_vecrecip.h                                             */
/* ------------------------------------------------------------------------ */
/*            Copyright (c) 2004 Texas Instruments, Incorporated.           */
/*                           All Rights Reserved.                           */
/* ======================================================================== */
