;==========================================================================;
;=INFORMATION           PROGRAM         ; PC controlled octave  generator =;
;=============          FILE NAME       ; pc_tune.asm                     =;
;=                      INCLUDE FILES   ; comm.asm, comm_init.asm         =;
;=                      PROCESSOR       ; TMS320C5x                       =;
;=                      AUTHOR          ; irm uh                          =;
;=                      DATE/VERSION    ; 4/97  / Ver 1.00                =;
;==========================================================================;
;=DESCRIPTION                                                             =;
;=============                                                            =;
;=                                                                        =;
;= pc_tune:   A program that initialises the TMS320C50 DSK processor      =;
;=            and AIC codec. At a sampling rate of 8kHz, the DSK          =;
;=            performs a serial inpu port dummy read.                     =;
;=                                                                        =;
;=            A digital oscillator program generates a series of sine     =;
;=            wave constituting an octave                                 =;
;=                                                                        =;
;=            no harmonics are used only pure sine waves                  =;
;=            The frequencies of the sine waves can be altered by sending =;
;=            different values via the rs232 port                         =;
;=                                                                        =;
;=                                                                        =;
;==========================================================================;
;=SPECIFICATION         FREQUENCY       ; 8kHz                            =;
;===============        FILTER          ; not present                     =;
;=                      MISC            ; TLC320C40 initialised           =;
;==========================================================================;
	.mmregs
;==========================================================================;
;=SETTINGS                                                                =;
;==========================================================================;
	.ds     1000h           ; Assemble to data memory
	.data
TA      .word   17              ; TA AIC value - Fcut = 8 KHz
RA      .word   17              ; RA AIC value - Fcut = 8 KHz
TB      .word   37              ; TB AIC value - Fs = 2*Fcut
RB      .word   37              ; RB AIC value - Fs = 2*Fcut
AIC_CTR .word   8h              ; |LP xx G1 G0 | SY AX LB BP|
				; +------------+------------+
				;  |     GAIN    |  |  |  +-- BP Filter
				;  |     Synch --+  |  +----- Loopback
				;  |     Auxin -----+
				;  + (sinx)/x filter
;-------------------------------------------;
; note table
; fn = f note/f sampling
; Y/2 in Q15 format
; A/2 = cos(fn*360) in Q15 format
;-------------------------------------------;


;= This file contains the time constants required for RS232 communication
 .include "com_init.asm"

do                      ;880Hz
		.word   10453     ; Initial Y =  [sin(fn*360)/2] Q15
		.word   0         ; Secondary Y value
docoef
		.word   25264     ; cos(fn*360) in Q15
re
		.word    11469    ; 987Hz
		.word     0
recoef
		 .word    23396   ; coeff value
mi
		 .word   11993    ;1046Hz
		 .word    0
micoef
		 .word   22315    ;coeff
fa
		 .word   13074    ; 1174Hz
		 .word   0
facoef
		 .word   19792
sol
		 .word     14090  ; 1318Hz
		 .word     0
solcoef
		 .word    16712
la
		 .word   14582    ;1397Hz
		 .word    0
lacoef
		 .word   14942
si
		 .word   15466    ; 1568Hz
		 .word    0
sicoef
		 .word   10912
;---------------------------------------;
;-- end of note table----------------;

;- temp variable to hold a character from the RS232 port
TEMP     .word 0

;==========================================================================;
;=INTERRUPT SETTINGS                                                      =;
;==========================================================================;
	.ps     080ah           ; Assemble to program memory
RINT    b       RX              ; Branch to RX on receive interrupt
XINT    b       TX              ; Branch to TX on transmit interrupt
;==========================================================================;
;=PROCESSOR INITIALISATION                                                =;
;==========================================================================;
	.ps 0a00h               ; Assemble to program memory
	.entry                  ; Mark program entry point
	.text                   ; Text
START   setc    INTM            ; Disable interrupts
	ldp     #0              ; Set data page pointer to page zero
	splk    #0836h,PMST     ; Write 16 bit pattern to PMST register
	lacc    #0              ; Load accumulator with number zero
	;-----------------------;
	samm    CWSR            ; Set software wait state to zero
	samm    PDWSR           ; Set software wait state to zero
; DISABLE KERNEL
	splk    #020h,IMR       ; Using XINT syn TX & RX
	call    AIC             ; Initialize AIC and enable interrupts
	;-----------------------;
	clrc    OVM             ; Overflow mode is set to zero
	spm     0                 ; Product shift mode is set to zero
	clrc   SXM
	lar      ar1,#do          ;point to first note
	lar      ar2,#docoef    ;point to first coefficient
	

	clrc    INTM            ; Enable interrupts

;==========================================================================;
;=MAIN PROGRAM                                                            =;
;==========================================================================;

WAIT
	  nop
	  nop
	  nop
;- Wait for a new character
	  call  RXBYTE
;- and store it somewhere safe
	  sacl  TEMP


;- choose a note to play depending on the value sent from the RS232
;- The program will only respond to the numbers 1 to 7 where
;- 1 represents do and 7 represents si

setdo lacl TEMP
		sub #1,0
		bcnd setre,NEQ
		lar  ar1,#do
		lar  ar2,#docoef       ; set DO
		b WAIT

setre   lacl TEMP
		sub #2,0
		bcnd setmi,NEQ
		lar  ar1,#re            ;point to RE in note table
		lar  ar2,#recoef        ; point to RE coefficient in note table
		b WAIT

setmi lacl TEMP
		sub #3,0
		bcnd setfa,NEQ
		lar  ar1,#mi            ;point aux regs to MI
		lar  ar2,#micoef
		b WAIT

setfa lacl TEMP
		sub #4,0
		bcnd setsol,NEQ
		lar  ar1,#fa            ;point aux regs to FA
		lar  ar2,#facoef
		b WAIT

setsol lacl TEMP
		 sub #5,0
		 bcnd setla,NEQ
		 lar  ar1,#sol          ;point aux regs to SOL
		 lar  ar2,#solcoef
		 b WAIT

setla  lacl TEMP
		 sub #6,0
		 bcnd setsi,NEQ
		 lar  ar1,#la           ;point aux regs to LA
		 lar  ar2,#lacoef
		 b WAIT

setsi  lacl TEMP
		 sub #7,0
		 bcnd WAIT,NEQ
		 lar  ar1,#si           ;point aux regs to SI
		 lar  ar2,#sicoef
		 b WAIT

 b       WAIT            ; Branch to WAIT

;---- end of main program ------;

;==========================================================================;
;= TRANSMIT INTERRUPT SERVICE ROUTINE                                     =;
;==========================================================================;

TX
;--------------------------------;
; note generation
; ar1 --> Y, ar2 --> coeff
; ar1 moves from Y to Y1
;--------------------------------;
	zap                     ; Clear acc and product register
	;-----------------------;
	; acc = -B*Y1
	;----------------------;
	mar  *,ar1              ;make ar1 current
	mar  *+,ar1             ;point to next value Y1
	lacc   *-,15,ar2        ; Load acc with Y1 with shift of 15,ar1->Y
	neg                     ; Negate accumulator
	;-----------------------;
	;  treg0 = coef
	;  ar1 --> Y
	;  preg = coef*Y
	;-----------------------;
	lt     *,ar1            ;load treg0=coeff, make ar1 current
	mpy    *                ;coeff*Y & place in preg, ar1 current
	dmov   *                ;Y->Y1 for next time round
	apac                    ; Add product register to acc
	apac                    ; all coeffs halved
	sach   *,1               ; out of Q15
	;-----------------------;
	lacc   *                ;load sample to send
	ldp     #0              ; load data page zero
	and     #0FFFCh         ; Clear lower two (status) bit from acc
	samm    DXR             ; Write lower acc to DXR
	mar *,AR5
	rete                    ; Return to main program

;==========================================================================;
;=RECEIVE INTERRUPT SERVICE ROUTINE                                       =;
;==========================================================================;
RX
	rete                    ; RX INT not used
;==========================================================================;
;=BOARD INITIALISATION                                                    =;
;==========================================================================;
AIC     splk    #20h,TCR        ; To generate 10 MHz from Tout
	splk    #01h,PRD        ; Load period counter with 1
	mar     *,AR0           ; Modify AR pointer (used with GREG)
	lacc    #0008h          ; Load acc with 08h
	sacl    SPC             ; Store acc in SPC
	lacc    #00C8h          ; Load acc with 0C8h
	sacl    SPC             ; Set and reset SPC register
	lacc    #080h           ; Initialise 8000h to FFFFh as global memory
	sach    DXR             ; Store high acc to DXR
	sacl    GREG            ; Store to global memory register
	lar     AR0,#0FFFFh     ; Set AR0 register
	rpt     #10000          ; Set repeat counter
	lacc    *,0,AR0         ; Access global memory
	sach    GREG            ; Disable global memory
	;-----------------------;
	ldp     #TA             ; Load data page TA
	setc    SXM             ; Set SXM for sign extension mode
	lacc    TA,9            ; Load accumulator with TA data
	add     RA,2            ; Load accumulator with RB data
	call    AIC2ND          ; Call function AIC2ND
	;-----------------------;
	ldp     #TB             ; Load data page TB
	lacc    TB,9            ; Load accumulator with TB data
	add     RB,2            ; Load accumulator with RB data
	add     #02h            ; Add 10b to set status bits
	call    AIC2ND          ; Call function AIC2ND
	;-----------------------;
	ldp     #AIC_CTR        ; Load data page AIC_CTR
	lacc    AIC_CTR,2       ; Initialized control register
	add     #03h            ; Add 11b to set status bits
	call    AIC2ND          ; Call function AIC2ND
	ret                     ; Return from call
;-------------------------------;
AIC2ND  ldp     #0              ; Load data page zero
	sach    DXR             ; Store upper acc to DXR
	clrc    INTM            ; Enable interrupts
	idle                    ; Idle until interrupt - TINT
	add     #6h,15          ; Set 2 LSBs of upper acc high
	sach    DXR             ; Store upper acc to DXR
	idle                    ; Idle until interrupt - TINT
	sacl    DXR             ; Store upper acc to DXR
	idle                    ; Idle until interrupt - TINT
	lacl    #0              ; Store lower acc to DXR - flushing
	sacl    DXR             ; make sure the word got sent
	idle                    ; Idle until interrupt - TINT
	setc    INTM            ; Disable interrupts
	ret                     ; Return from call
;-------------------------------;

;= Include the RS232 Comm routines
 .include "comm.asm"
	.end                    ; End of program






